import { Input, Number, Select, SelectAsync, Sort, Toggle, CollapsibleContent, NestedRepeater } from 'modules/UI';
import { useListingData, useLocalizedData, useAsyncOptions } from "services/data";

const { postTypeOptions, postsOrderByOptions, postStatiOptions, taxonomiesOptions, taxonomyTermFieldOptions, termCompareOperatorOptions, metaCompareOperatorOptions, metaTypeOptions } = useLocalizedData();
const { loadPostOptions, loadUserOptions } = useAsyncOptions();

const QuerySettings = () => {
	// Data
	const {
// selectors
		listingQueryProp: queryProp,
		// actions
		setListingQueryProp: updateQueryProp,
	} = useListingData();

	// Actions
	const updateRepeaterItem = (newValue, settingKey, item, repeaterKey) => {
		item[settingKey] = newValue;
		updateQueryProp(repeaterKey, queryProp(repeaterKey));
	};

	const clausesButtonLabel = (count) => {
		return count > 0 ? `افزودن گزاره OR` : 'افزودن گزاره‌ها';
	}

	return (
		<div className="jsf-listings-edit__content">
			<h2>محتوا</h2>
			<div className="jsf-control-posts-count">
				<Number
					label="در هر صفحه"
					min='1'
					value={queryProp('posts_per_page')}
					onChange={(newValue) => updateQueryProp('posts_per_page', newValue)}
				/>
				<Number
					label="جابه‌جایی (Offset)"
					min='0'
					value={queryProp('offset')}
					onChange={(newValue) => updateQueryProp('offset', newValue)}
				/>
			</div>
			<div className="jsf-listings-edit__content__separator"></div>
			<h2>ترتیب</h2>
			<Sort
				value={queryProp('sort')}
				options={postsOrderByOptions}
				onChange={(newValue) => updateQueryProp('sort', newValue)}
			/>
			<div className="jsf-listings-edit__content__separator"></div>
			<h2>نوشته‌ها</h2>
			<Select
				label="انواع نوشته‌ها"
				placeholder="نوشته‌ها"
				value={queryProp('post_types')}
				options={postTypeOptions}
				isMulti={true}
				onChange={(newValue) => updateQueryProp('post_types', newValue)}
			/>
			<Select
				label="وضعیت نوشته"
				placeholder="منتشرشده"
				value={queryProp('post_status')}
				options={postStatiOptions}
				isMulti={true}
				onChange={(newValue) => updateQueryProp('post_status', newValue)}
			/>
			<SelectAsync
				label="نویسندگان نوشته‌ها"
				placeholder="هر نویسنده"
				value={queryProp('post_authors')}
				loadOptions={loadUserOptions}
				isMulti={true}
				onChange={(newValue) => updateQueryProp('post_authors', newValue)}
			/>
			<SelectAsync
				label="افزودن نوشته‌ها"
				placeholder="هیچ‌کدام"
				value={queryProp('post__in')}
				loadOptions={loadPostOptions}
				isMulti={true}
				onChange={(newValue) => updateQueryProp('post__in', newValue)}
			/>
			<SelectAsync
				label="حذف نوشته‌ها"
				placeholder="هیچ‌کدام"
				value={queryProp('post__not_in')}
				loadOptions={loadPostOptions}
				isMulti={true}
				onChange={(newValue) => updateQueryProp('post__not_in', newValue)}
			/>
			<Toggle
				label="نادیده‌گرفتن نوشته‌های سنجاق‌شده"
				value={queryProp('ignore_sticky_posts')}
				onChange={(newValue) => updateQueryProp('ignore_sticky_posts', newValue)}
			/>
			<CollapsibleContent
				closeLabel="نمایش تنظیمات پیشرفته"
				openLabel="پنهان‌سازی تنظیمات پیشرفته"
			>
				<h2>تاکسونومی‌ها</h2>
				<p>نمایش محتوای مربوط به تاکسونومی‌های خاص.</p>
				<NestedRepeater
					items={queryProp('taxonomies')}
					defaultItemData={{
						taxonomy: '',
						term_field: 'term_id',
						terms: '',
						operator: 'IN',
						child_terms: false,
						isOpen: true
					}}
					externalSeparator={<div className="jsf-repeater__separator__or">Or</div>}
					externalButtonAddLabel={ clausesButtonLabel( queryProp('taxonomies').length ) }
				internalLabelMask="Taxonomy:{taxonomy::_} {term_field}(s):{terms::_} operator:{operator}"
					internalSeparator={<div className="jsf-repeater__separator__and">And</div>}
					internalButtonAddLabel="افزودن گزاره AND"
					onChange={(newValue) => updateQueryProp('taxonomies', newValue)}
				>
					{(item, index) =>
						<div className="jsf-taxonomies-repeater-item">
							<Select
								label="تاکسونومی"
								description="تاکسونومی را برای دریافت نوشته‌ها انتخاب کنید"
								value={item.taxonomy}
								options={taxonomiesOptions}
								onChange={(newValue) => updateRepeaterItem(newValue, 'taxonomy', item, 'taxonomies')}
							/>
							<Select
								label="فیلد"
								description="انتخاب واژه تاکسونومی بر اساس"
								value={item.term_field}
								options={taxonomyTermFieldOptions}
								onChange={(newValue) => updateRepeaterItem(newValue, 'term_field', item, 'taxonomies')}
							/>
							<Input
								label="واژه‌ها"
								description="واژه(های) تاکسونومی برای دریافت نوشته‌ها"
								value={item.terms}
								onChange={(newValue) => updateRepeaterItem(newValue, 'terms', item, 'custom_fields')}
							/>
							<Select
								label="عملگر مقایسه"
								description="عملگر برای مقایسه واژه‌ها"
								value={item.operator}
								options={termCompareOperatorOptions}
								onChange={(newValue) => updateRepeaterItem(newValue, 'operator', item, 'taxonomies')}
							/>
							<Toggle
								label="واژه‌های فرزند"
								description="شامل واژه‌های زیرمجموعه در تاکسونومی‌های سلسله‌مراتبی."
								value={item.child_terms}
								onChange={(newValue) => updateRepeaterItem(newValue, 'child_terms', item, 'taxonomies')}
							/>
						</div>
					}
				</NestedRepeater>
				<div className="jsf-listings-edit__content__separator"></div>
				<h2>فیلدهای سفارشی</h2>
				<p>نمایش محتوای مربوط به فیلدهای سفارشی خاص.</p>
				<NestedRepeater
					items={queryProp('custom_fields')}
					defaultItemData={{
						key: '',
						type: 'CHAR',
						compare: '=',
						value: '',
						isOpen: true
					}}
					externalSeparator={<div className="jsf-repeater__separator__or">Or</div>}
					externalButtonAddLabel={ clausesButtonLabel( queryProp('custom_fields').length ) }
					internalLabelMask="Field '{key::_}' {compare} {value::_}({type})"
					internalSeparator={<div className="jsf-repeater__separator__and">And</div>}
					internalButtonAddLabel="Add AND Clause"
					onChange={(newValue) => updateQueryProp('custom_fields', newValue)}
				>
					{(item, index) =>
						<div className="jsf-custom-fields-repeater-item">
							<div className="jsf-custom-fields-repeater-item__row">
								<Input
									label="کلید فیلد"
									value={item.key}
									onChange={(newValue) => updateRepeaterItem(newValue, 'key', item, 'custom_fields')}
								/>
								<Select
									label="نوع فیلد"
									value={item.type}
									options={metaTypeOptions}
									onChange={(newValue) => updateRepeaterItem(newValue, 'type', item, 'custom_fields')}
								/>
							</div>
							<div className="jsf-custom-fields-repeater-item__row">
								<Select
									label="مقایسه با"
									value={item.compare}
									options={metaCompareOperatorOptions}
									onChange={(newValue) => updateRepeaterItem(newValue, 'compare', item, 'custom_fields')}
								/>
								<Input
									label="مقدار فیلد"
									value={item.value}
									onChange={(newValue) => updateRepeaterItem(newValue, 'value', item, 'custom_fields')}
								/>
							</div>
						</div>
					}
				</NestedRepeater>
			</CollapsibleContent>
		</div>
	);
};

export default QuerySettings;
